"use strict";
(() => {
  // src/shared/messages.ts
  var BG_MSG = {
    FETCH: "FRANZAI_FETCH",
    FETCH_ABORT: "FRANZAI_FETCH_ABORT",
    GET_SETTINGS: "FRANZAI_GET_SETTINGS",
    SET_SETTINGS: "FRANZAI_SET_SETTINGS",
    GET_LOGS: "FRANZAI_GET_LOGS",
    CLEAR_LOGS: "FRANZAI_CLEAR_LOGS"
  };
  var BG_EVT = {
    LOGS_UPDATED: "FRANZAI_LOGS_UPDATED",
    SETTINGS_UPDATED: "FRANZAI_SETTINGS_UPDATED"
  };

  // src/shared/logger.ts
  var LEVELS = {
    debug: 10,
    info: 20,
    warn: 30,
    error: 40,
    silent: 99
  };
  function resolveLevel(level) {
    if (level) return level;
    const raw = globalThis.__FRANZAI_LOG_LEVEL__;
    if (typeof raw === "string" && raw in LEVELS) return raw;
    return "info";
  }
  function createLogger(scope, level, c = console) {
    const chosen = resolveLevel(level);
    const min = LEVELS[chosen];
    const prefix = `[FranzAI Bridge/${scope}]`;
    return {
      debug: (...args) => {
        if (min <= LEVELS.debug) c.debug(prefix, ...args);
      },
      info: (...args) => {
        if (min <= LEVELS.info) c.info(prefix, ...args);
      },
      warn: (...args) => {
        if (min <= LEVELS.warn) c.warn(prefix, ...args);
      },
      error: (...args) => {
        if (min <= LEVELS.error) c.error(prefix, ...args);
      },
      log: (...args) => {
        if (min <= LEVELS.info) c.log(prefix, ...args);
      }
    };
  }

  // src/shared/constants.ts
  var BRIDGE_VERSION = "2.0.27";
  var MAX_BODY_BYTES = 5 * 1024 * 1024;
  var RUNTIME_MESSAGE_TIMEOUT_MS = 15e3;

  // src/shared/runtime.ts
  var log = createLogger("runtime");
  function sendRuntimeMessage(message, options = {}) {
    const timeoutMs = options.timeoutMs ?? RUNTIME_MESSAGE_TIMEOUT_MS;
    return new Promise((resolve, reject) => {
      let done = false;
      const timeoutId = window.setTimeout(() => {
        if (done) return;
        done = true;
        reject(new Error(`Timeout waiting for runtime response after ${timeoutMs}ms`));
      }, timeoutMs);
      try {
        chrome.runtime.sendMessage(message, (resp) => {
          if (done) return;
          done = true;
          clearTimeout(timeoutId);
          const err = chrome.runtime.lastError;
          if (err) {
            log.warn("sendMessage failed", err.message || err);
            reject(err);
            return;
          }
          resolve(resp);
        });
      } catch (e) {
        if (done) return;
        done = true;
        clearTimeout(timeoutId);
        log.error("sendMessage threw", e);
        reject(e);
      }
    });
  }

  // src/sidepanel/sidepanel.ts
  var log2 = createLogger("sidepanel");
  function qs(id) {
    const el = document.getElementById(id);
    if (!el) throw new Error(`Missing element #${id}`);
    return el;
  }
  var settings = null;
  var logs = [];
  var selectedLogId = null;
  var logsList = qs("logsList");
  var details = qs("details");
  var DEFAULT_COL_WIDTHS = [50, 45, 90, -1, 45, 40];
  var colWidths = [...DEFAULT_COL_WIDTHS];
  var sortColumn = "ts";
  var sortDir = "desc";
  var filterSearch = "";
  var filterMethod = "";
  var filterStatus = "";
  var UI_PREFS_KEY = "franzai_ui_prefs";
  function debounce(fn, ms) {
    let timeout;
    return ((...args) => {
      clearTimeout(timeout);
      timeout = setTimeout(() => fn(...args), ms);
    });
  }
  async function loadUIPrefs() {
    try {
      const data = await chrome.storage.local.get(UI_PREFS_KEY);
      const prefs = data[UI_PREFS_KEY];
      if (prefs) {
        if (prefs.colWidths) colWidths = prefs.colWidths;
        if (prefs.sortColumn) sortColumn = prefs.sortColumn;
        if (prefs.sortDir) sortDir = prefs.sortDir;
      }
    } catch (e) {
      log2.error("Failed to load UI preferences", e);
    }
  }
  var saveUIPrefs = debounce(async () => {
    try {
      await chrome.storage.local.set({
        [UI_PREFS_KEY]: { colWidths, sortColumn, sortDir }
      });
    } catch (e) {
      log2.error("Failed to save UI preferences", e);
    }
  }, 500);
  function hasActiveFilters() {
    return filterSearch !== "" || filterMethod !== "" || filterStatus !== "";
  }
  function clearFilters() {
    filterSearch = "";
    filterMethod = "";
    filterStatus = "";
    const searchInput = document.getElementById("filterSearch");
    const methodSelect = document.getElementById("filterMethod");
    const statusSelect = document.getElementById("filterStatus");
    if (searchInput) searchInput.value = "";
    if (methodSelect) methodSelect.value = "";
    if (statusSelect) statusSelect.value = "";
    renderLogs();
  }
  function fmtTs(ts) {
    const d = new Date(ts);
    return d.toLocaleString();
  }
  function fmtShortTime(ts) {
    const d = new Date(ts);
    return d.toLocaleTimeString([], { hour: "2-digit", minute: "2-digit", second: "2-digit" });
  }
  function escapeHtml(s) {
    return s.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;").replaceAll('"', "&quot;").replaceAll("'", "&#039;");
  }
  function getStatusClass(status, error) {
    if (error) return "error";
    if (!status) return "";
    if (status >= 200 && status < 300) return "success";
    if (status >= 300 && status < 400) return "redirect";
    if (status >= 400) return "error";
    return "";
  }
  function highlightJson(json) {
    const escaped = escapeHtml(json);
    return escaped.replace(/"([^"\\]*(\\.[^"\\]*)*)"/g, (match, content) => {
      return `<span class="json-string">"${content}"</span>`;
    }).replace(/\b(-?\d+\.?\d*)\b/g, '<span class="json-number">$1</span>').replace(/\b(true|false|null)\b/g, '<span class="json-bool">$1</span>');
  }
  function createLogItem(l) {
    const div = document.createElement("div");
    div.className = "item" + (l.id === selectedLogId ? " active" : "");
    const status = l.error ? "ERR" : l.status ?? "...";
    const statusClass = getStatusClass(l.status, l.error);
    const ms = l.elapsedMs != null ? `${l.elapsedMs}` : "\u2014";
    let host = "";
    let path = l.url;
    try {
      const u = new URL(l.url);
      host = u.host;
      path = u.pathname + u.search;
      if (path.length > 40) {
        path = path.substring(0, 37) + "...";
      }
    } catch {
    }
    const tsDiv = document.createElement("div");
    tsDiv.className = "ts";
    tsDiv.textContent = fmtShortTime(l.ts);
    tsDiv.title = fmtTs(l.ts);
    div.appendChild(tsDiv);
    const methodDiv = document.createElement("div");
    methodDiv.className = "method " + l.method;
    methodDiv.textContent = l.method;
    div.appendChild(methodDiv);
    const hostDiv = document.createElement("div");
    hostDiv.className = "host";
    hostDiv.textContent = host;
    hostDiv.title = host;
    div.appendChild(hostDiv);
    const urlDiv = document.createElement("div");
    urlDiv.className = "url";
    urlDiv.textContent = path;
    urlDiv.title = l.url;
    div.appendChild(urlDiv);
    const statusDiv = document.createElement("div");
    statusDiv.className = "status-code " + statusClass;
    statusDiv.textContent = String(status);
    div.appendChild(statusDiv);
    const elapsedDiv = document.createElement("div");
    elapsedDiv.className = "elapsed";
    elapsedDiv.textContent = ms;
    div.appendChild(elapsedDiv);
    return div;
  }
  function closeDetailPane() {
    selectedLogId = null;
    const detailPane = document.getElementById("detailPane");
    if (detailPane) detailPane.classList.remove("visible");
    document.querySelectorAll(".item.active").forEach((el) => el.classList.remove("active"));
  }
  function filterLogs(logsToFilter) {
    return logsToFilter.filter((l) => {
      if (filterSearch && !l.url.toLowerCase().includes(filterSearch.toLowerCase())) {
        return false;
      }
      if (filterMethod && l.method !== filterMethod) {
        return false;
      }
      if (filterStatus) {
        const status = l.status ?? 0;
        if (filterStatus === "success" && (status < 200 || status >= 300)) return false;
        if (filterStatus === "redirect" && (status < 300 || status >= 400)) return false;
        if (filterStatus === "error" && status < 400 && !l.error) return false;
      }
      return true;
    });
  }
  function getHost(url) {
    try {
      return new URL(url).host;
    } catch {
      return "";
    }
  }
  function sortLogs(logsToSort) {
    const sorted = [...logsToSort];
    const dir = sortDir === "asc" ? 1 : -1;
    sorted.sort((a, b) => {
      let cmp = 0;
      switch (sortColumn) {
        case "ts":
          cmp = a.ts - b.ts;
          break;
        case "method":
          cmp = a.method.localeCompare(b.method);
          break;
        case "host":
          cmp = getHost(a.url).localeCompare(getHost(b.url));
          break;
        case "url":
          cmp = a.url.localeCompare(b.url);
          break;
        case "status":
          cmp = (a.status ?? 0) - (b.status ?? 0);
          break;
        case "elapsed":
          cmp = (a.elapsedMs ?? 0) - (b.elapsedMs ?? 0);
          break;
      }
      return cmp * dir;
    });
    return sorted;
  }
  function updateSortIndicators() {
    document.querySelectorAll(".table-header .sortable").forEach((el) => {
      el.classList.remove("asc", "desc");
      if (el.getAttribute("data-col") === sortColumn) {
        el.classList.add(sortDir);
      }
    });
  }
  function renderLogs() {
    const fragment = document.createDocumentFragment();
    logsList.innerHTML = "";
    const detailPane = document.getElementById("detailPane");
    if (!logs.length) {
      const hint = document.createElement("div");
      hint.className = "hint";
      hint.textContent = "No requests captured yet. Make fetch calls to see them here.";
      logsList.appendChild(hint);
      if (detailPane) detailPane.classList.remove("visible");
      updateFilterUI(0, 0);
      return;
    }
    const filteredLogs = filterLogs(logs);
    const sortedLogs = sortLogs(filteredLogs);
    updateFilterUI(filteredLogs.length, logs.length);
    if (!filteredLogs.length) {
      const hint = document.createElement("div");
      hint.className = "hint";
      hint.innerHTML = `No requests match your filters. <a href="#" id="clearFiltersLink">Clear filters</a>`;
      logsList.appendChild(hint);
      const clearLink = document.getElementById("clearFiltersLink");
      if (clearLink) clearLink.onclick = (e) => {
        e.preventDefault();
        clearFilters();
      };
      if (detailPane) detailPane.classList.remove("visible");
      return;
    }
    for (const l of sortedLogs) {
      const div = createLogItem(l);
      div.onclick = () => {
        if (selectedLogId === l.id) {
          closeDetailPane();
          return;
        }
        selectedLogId = l.id;
        document.querySelectorAll(".item.active").forEach((el) => el.classList.remove("active"));
        div.classList.add("active");
        renderDetails(l);
        if (detailPane) detailPane.classList.add("visible");
      };
      fragment.appendChild(div);
    }
    logsList.appendChild(fragment);
    if (sortColumn === "ts" && sortDir === "desc") {
      logsList.scrollTop = logsList.scrollHeight;
    }
    updateSortIndicators();
    if (selectedLogId && detailPane) {
      detailPane.classList.add("visible");
    }
  }
  function updateFilterUI(filtered, total) {
    const countEl = document.getElementById("requestCount");
    if (countEl) {
      countEl.textContent = filtered === total ? String(total) : `${filtered}/${total}`;
      countEl.classList.toggle("filtered", filtered !== total);
    }
    const clearBtn = document.getElementById("btnClearFilters");
    if (clearBtn) {
      clearBtn.style.display = hasActiveFilters() ? "inline-flex" : "none";
    }
  }
  function copyToClipboard(text, btn) {
    navigator.clipboard.writeText(text).then(() => {
      const original = btn.textContent;
      btn.textContent = "Copied!";
      btn.classList.add("copied");
      setTimeout(() => {
        btn.textContent = original;
        btn.classList.remove("copied");
      }, 1e3);
    });
  }
  function createSection(title, content, id, useHighlight = true) {
    const section = document.createElement("div");
    section.className = "detail-section";
    const header = document.createElement("div");
    header.className = "section-header";
    const titleEl = document.createElement("span");
    titleEl.className = "section-title";
    titleEl.textContent = title;
    const copyBtn = document.createElement("button");
    copyBtn.className = "copy-btn";
    copyBtn.textContent = "Copy";
    copyBtn.onclick = (e) => {
      e.stopPropagation();
      copyToClipboard(content, copyBtn);
    };
    header.appendChild(titleEl);
    header.appendChild(copyBtn);
    const pre = document.createElement("pre");
    pre.id = id;
    if (useHighlight && (content.startsWith("{") || content.startsWith("["))) {
      pre.innerHTML = highlightJson(content);
    } else {
      pre.textContent = content;
    }
    section.appendChild(header);
    section.appendChild(pre);
    return section;
  }
  function renderDetails(l) {
    details.innerHTML = "";
    const topBar = document.createElement("div");
    topBar.className = "detail-top-bar";
    const copyUrlBtn = document.createElement("button");
    copyUrlBtn.className = "copy-url-btn";
    copyUrlBtn.textContent = "Copy URL";
    copyUrlBtn.onclick = () => copyToClipboard(l.url, copyUrlBtn);
    const copyAllBtn = document.createElement("button");
    copyAllBtn.className = "copy-all-btn";
    copyAllBtn.textContent = "Copy All";
    const closeBtn = document.createElement("button");
    closeBtn.className = "close-detail-btn";
    closeBtn.innerHTML = "\u2715";
    closeBtn.title = "Close (Esc)";
    closeBtn.onclick = () => closeDetailPane();
    copyAllBtn.onclick = () => {
      const allData = {
        request: {
          requestId: l.requestId,
          ts: fmtTs(l.ts),
          tabId: l.tabId,
          pageOrigin: l.pageOrigin,
          method: l.method,
          url: l.url,
          headers: l.requestHeaders,
          body: l.requestBodyPreview || void 0
        },
        response: {
          status: l.status,
          statusText: l.statusText,
          elapsedMs: l.elapsedMs,
          error: l.error || void 0,
          headers: l.responseHeaders,
          body: l.responseBodyPreview || void 0
        }
      };
      copyToClipboard(JSON.stringify(allData, null, 2), copyAllBtn);
    };
    topBar.appendChild(copyUrlBtn);
    topBar.appendChild(copyAllBtn);
    topBar.appendChild(closeBtn);
    details.appendChild(topBar);
    const requestData = JSON.stringify({
      method: l.method,
      url: l.url,
      headers: l.requestHeaders
    }, null, 2);
    details.appendChild(createSection("Request Headers", requestData, "req-headers"));
    if (l.requestBodyPreview && l.requestBodyPreview.trim()) {
      details.appendChild(createSection("Request Body", l.requestBodyPreview, "req-body"));
    }
    const responseData = JSON.stringify({
      status: l.status,
      statusText: l.statusText,
      elapsedMs: l.elapsedMs,
      error: l.error || void 0,
      headers: l.responseHeaders
    }, null, 2);
    details.appendChild(createSection("Response Headers", responseData, "resp-headers"));
    if (l.responseBodyPreview && l.responseBodyPreview.trim()) {
      details.appendChild(createSection("Response Body", l.responseBodyPreview, "resp-body"));
    }
    const metaData = JSON.stringify({
      requestId: l.requestId,
      ts: fmtTs(l.ts),
      tabId: l.tabId,
      pageOrigin: l.pageOrigin
    }, null, 2);
    details.appendChild(createSection("Meta", metaData, "meta"));
  }
  function showToast(message, isError = false) {
    const existing = document.querySelector(".toast");
    if (existing) existing.remove();
    const toast = document.createElement("div");
    toast.className = "toast" + (isError ? " error" : "");
    toast.textContent = message;
    document.body.appendChild(toast);
    setTimeout(() => toast.classList.add("visible"), 10);
    setTimeout(() => {
      toast.classList.remove("visible");
      setTimeout(() => toast.remove(), 300);
    }, 2500);
  }
  var ICON_EDIT = `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/></svg>`;
  var ICON_TRASH = `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="3 6 5 6 21 6"/><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/></svg>`;
  function renderListTable(config) {
    const table = qs(config.tableId);
    table.innerHTML = "";
    if (!config.items.length) {
      const hint = document.createElement("div");
      hint.className = "hint";
      hint.textContent = config.emptyText;
      table.appendChild(hint);
      return;
    }
    for (const v of config.items) {
      const tr = document.createElement("div");
      tr.className = "tr";
      const tdValue = document.createElement("div");
      tdValue.className = "td";
      tdValue.textContent = v;
      const tdActions = document.createElement("div");
      tdActions.className = "td actions";
      if (config.onEdit) {
        const editBtn = document.createElement("button");
        editBtn.className = "icon-btn";
        editBtn.innerHTML = ICON_EDIT;
        editBtn.title = "Edit";
        editBtn.onclick = () => startInlineEdit(tr, tdValue, v, config.onEdit);
        tdActions.appendChild(editBtn);
      }
      const delBtn = document.createElement("button");
      delBtn.className = "icon-btn";
      delBtn.innerHTML = ICON_TRASH;
      delBtn.title = "Delete";
      delBtn.onclick = () => config.onDelete(v);
      tdActions.appendChild(delBtn);
      tr.appendChild(tdValue);
      tr.appendChild(tdActions);
      table.appendChild(tr);
    }
  }
  function startInlineEdit(row, cell, currentValue, onSave) {
    const input = document.createElement("input");
    input.type = "text";
    input.value = currentValue;
    input.className = "inline-edit";
    const originalContent = cell.textContent;
    cell.textContent = "";
    cell.appendChild(input);
    input.focus();
    input.select();
    const save = async () => {
      const newValue = input.value.trim();
      if (newValue && newValue !== currentValue) {
        await onSave(currentValue, newValue);
      } else {
        cell.textContent = originalContent;
      }
    };
    input.onblur = save;
    input.onkeydown = (e) => {
      if (e.key === "Enter") save();
      if (e.key === "Escape") {
        cell.textContent = originalContent;
      }
    };
  }
  function renderEnvTable() {
    const envTable = qs("envTable");
    envTable.innerHTML = "";
    const env = settings?.env ?? {};
    const keys = Object.keys(env).sort((a, b) => a.localeCompare(b));
    if (!keys.length) {
      const hint = document.createElement("div");
      hint.className = "hint";
      hint.textContent = "No ENV vars configured.";
      envTable.appendChild(hint);
      return;
    }
    for (const k of keys) {
      const value = env[k];
      const hasValue = value && value.trim() !== "";
      const tr = document.createElement("div");
      tr.className = "tr env-row";
      const tdKey = document.createElement("div");
      tdKey.className = "td env-key";
      tdKey.textContent = k;
      const tdValue = document.createElement("div");
      tdValue.className = "td env-value";
      if (hasValue) {
        tdValue.textContent = "\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022";
        tdValue.title = "Value is set (hidden for security)";
      } else {
        tdValue.textContent = "Not set";
        tdValue.classList.add("not-set");
      }
      const tdActions = document.createElement("div");
      tdActions.className = "td actions";
      const editBtn = document.createElement("button");
      editBtn.className = "icon-btn";
      editBtn.innerHTML = ICON_EDIT;
      editBtn.title = hasValue ? "Edit" : "Set value";
      editBtn.onclick = () => showEnvEditModal(k, env[k] || "");
      tdActions.appendChild(editBtn);
      const delBtn = document.createElement("button");
      delBtn.className = "icon-btn";
      delBtn.innerHTML = ICON_TRASH;
      delBtn.title = "Delete";
      delBtn.onclick = async () => {
        if (!settings) return;
        const next = structuredClone(settings);
        delete next.env[k];
        await saveSettings(next);
        showToast(`Deleted ${k}`);
      };
      tdActions.appendChild(delBtn);
      tr.appendChild(tdKey);
      tr.appendChild(tdValue);
      tr.appendChild(tdActions);
      envTable.appendChild(tr);
    }
  }
  function showEnvEditModal(key, currentValue) {
    const overlay = document.createElement("div");
    overlay.className = "modal-overlay";
    const modal = document.createElement("div");
    modal.className = "modal";
    const title = document.createElement("div");
    title.className = "modal-title";
    title.textContent = `Edit ${key}`;
    const input = document.createElement("input");
    input.type = "text";
    input.value = currentValue;
    input.placeholder = `Enter value for ${key}`;
    input.className = "modal-input";
    const btnRow = document.createElement("div");
    btnRow.className = "modal-buttons";
    const cancelBtn = document.createElement("button");
    cancelBtn.textContent = "Cancel";
    cancelBtn.onclick = () => overlay.remove();
    const saveBtn = document.createElement("button");
    saveBtn.className = "primary";
    saveBtn.textContent = "Save";
    saveBtn.onclick = async () => {
      if (!settings) return;
      const next = structuredClone(settings);
      next.env[key] = input.value;
      await saveSettings(next);
      overlay.remove();
      showToast(`Updated ${key}`);
    };
    btnRow.appendChild(cancelBtn);
    btnRow.appendChild(saveBtn);
    modal.appendChild(title);
    modal.appendChild(input);
    modal.appendChild(btnRow);
    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    input.focus();
    input.select();
    const onKey = (e) => {
      if (e.key === "Escape") {
        overlay.remove();
        document.removeEventListener("keydown", onKey);
      }
      if (e.key === "Enter") {
        saveBtn.click();
      }
    };
    document.addEventListener("keydown", onKey);
    overlay.onclick = (e) => {
      if (e.target === overlay) overlay.remove();
    };
  }
  function renderOriginsTable() {
    renderListTable({
      tableId: "originsTable",
      items: settings?.allowedOrigins ?? [],
      emptyText: "No origins allowed (everything blocked).",
      onDelete: async (v) => {
        if (!settings) return;
        const next = structuredClone(settings);
        next.allowedOrigins = next.allowedOrigins.filter((x) => x !== v);
        await saveSettings(next);
        showToast("Origin removed");
      },
      onEdit: async (oldVal, newVal) => {
        if (!settings) return;
        const next = structuredClone(settings);
        const idx = next.allowedOrigins.indexOf(oldVal);
        if (idx >= 0) next.allowedOrigins[idx] = newVal;
        await saveSettings(next);
        showToast("Origin updated");
      }
    });
  }
  function renderDestsTable() {
    renderListTable({
      tableId: "destsTable",
      items: settings?.allowedDestinations ?? [],
      emptyText: "No destinations allowed (everything blocked).",
      onDelete: async (v) => {
        if (!settings) return;
        const next = structuredClone(settings);
        next.allowedDestinations = next.allowedDestinations.filter((x) => x !== v);
        await saveSettings(next);
        showToast("Destination removed");
      },
      onEdit: async (oldVal, newVal) => {
        if (!settings) return;
        const next = structuredClone(settings);
        const idx = next.allowedDestinations.indexOf(oldVal);
        if (idx >= 0) next.allowedDestinations[idx] = newVal;
        await saveSettings(next);
        showToast("Destination updated");
      }
    });
  }
  function renderRulesTable() {
    const table = qs("rulesTable");
    table.innerHTML = "";
    const items = settings?.injectionRules ?? [];
    if (!items.length) {
      table.innerHTML = `<div class="hint">No custom rules.</div>`;
      return;
    }
    items.forEach((r, idx) => {
      const tr = document.createElement("div");
      tr.className = "tr";
      const tdValue = document.createElement("div");
      tdValue.className = "td";
      tdValue.innerHTML = `
      <b>${escapeHtml(r.hostPattern)}</b><br />
      <span style="color: var(--muted)">headers:</span> ${escapeHtml(
        JSON.stringify(r.injectHeaders ?? {})
      )}<br />
      <span style="color: var(--muted)">query:</span> ${escapeHtml(
        JSON.stringify(r.injectQuery ?? {})
      )}
    `;
      const tdActions = document.createElement("div");
      tdActions.className = "td actions";
      const btn = document.createElement("button");
      btn.className = "icon-btn";
      btn.innerHTML = ICON_TRASH;
      btn.title = "Delete rule";
      btn.onclick = async () => {
        if (!settings) return;
        const next = structuredClone(settings);
        next.injectionRules.splice(idx, 1);
        await saveSettings(next);
        showToast("Rule deleted");
      };
      tdActions.appendChild(btn);
      tr.appendChild(tdValue);
      tr.appendChild(tdActions);
      table.appendChild(tr);
    });
  }
  function renderSettings() {
    renderEnvTable();
    renderOriginsTable();
    renderDestsTable();
    renderRulesTable();
  }
  async function loadAll() {
    log2.info("Loading...");
    try {
      const s = await sendRuntimeMessage({
        type: BG_MSG.GET_SETTINGS
      });
      if (s.ok && s.settings) settings = s.settings;
      const l = await sendRuntimeMessage({
        type: BG_MSG.GET_LOGS
      });
      if (l.ok && l.logs) logs = l.logs;
      log2.info(`Loaded ${logs.length} logs`);
    } catch (e) {
      log2.error("Failed to load state from background", e);
      showToast("Failed to load - check extension console", true);
    }
    renderLogs();
    renderSettings();
    if (selectedLogId && !logs.some((x) => x.id === selectedLogId)) {
      selectedLogId = null;
      details.innerHTML = `<div class="hint">Select a request to inspect.</div>`;
    }
  }
  async function saveSettings(next) {
    try {
      const resp = await sendRuntimeMessage({
        type: BG_MSG.SET_SETTINGS,
        payload: next
      });
      if (!resp.ok) {
        showToast(resp.error ?? "Failed to save settings", true);
        return;
      }
      settings = next;
      renderSettings();
      showToast("Settings saved");
    } catch (e) {
      log2.error("Failed to save settings", e);
      showToast("Failed to save settings", true);
    }
  }
  function parseJsonOrEmpty(txt) {
    if (!txt) return void 0;
    return JSON.parse(txt);
  }
  qs("btnRefresh").onclick = () => loadAll();
  qs("btnClearLogs").onclick = async () => {
    try {
      await sendRuntimeMessage({
        type: BG_MSG.CLEAR_LOGS
      });
      await loadAll();
      showToast("Logs cleared");
    } catch (e) {
      log2.error("Failed to clear logs", e);
      showToast("Failed to clear logs", true);
    }
  };
  qs("btnResetSettings").onclick = async () => {
    if (!confirm("Reset all settings to defaults? This will clear your API keys and custom rules.")) {
      return;
    }
    try {
      await chrome.storage.local.clear();
      await loadAll();
      showToast("Settings reset to defaults");
    } catch (e) {
      log2.error("Failed to reset settings", e);
      showToast("Failed to reset settings", true);
    }
  };
  qs("btnAddEnv").onclick = async () => {
    if (!settings) return;
    const nameInput = qs("envName");
    const valueInput = qs("envValue");
    const name = nameInput.value.trim();
    const value = valueInput.value;
    if (!name) {
      nameInput.classList.add("error");
      nameInput.focus();
      showToast("ENV name required", true);
      setTimeout(() => nameInput.classList.remove("error"), 2e3);
      return;
    }
    const next = structuredClone(settings);
    next.env[name] = value;
    nameInput.value = "";
    valueInput.value = "";
    await saveSettings(next);
    showToast(`Added ${name}`);
  };
  qs("btnAddOrigin").onclick = async () => {
    if (!settings) return;
    const input = qs("originValue");
    const v = input.value.trim();
    if (!v) {
      input.focus();
      return;
    }
    const next = structuredClone(settings);
    if (!next.allowedOrigins.includes(v)) {
      next.allowedOrigins.push(v);
      input.value = "";
      await saveSettings(next);
      showToast("Origin added");
    } else {
      showToast("Origin already exists", true);
    }
  };
  qs("btnAddDest").onclick = async () => {
    if (!settings) return;
    const input = qs("destValue");
    const v = input.value.trim();
    if (!v) {
      input.focus();
      return;
    }
    const next = structuredClone(settings);
    if (!next.allowedDestinations.includes(v)) {
      next.allowedDestinations.push(v);
      input.value = "";
      await saveSettings(next);
      showToast("Destination added");
    } else {
      showToast("Destination already exists", true);
    }
  };
  qs("btnAddRule").onclick = async () => {
    if (!settings) return;
    const hostInput = qs("ruleHost");
    const headersInput = qs("ruleHeaders");
    const queryInput = qs("ruleQuery");
    const hostPattern = hostInput.value.trim();
    const headersJson = headersInput.value.trim();
    const queryJson = queryInput.value.trim();
    if (!hostPattern) {
      hostInput.classList.add("error");
      hostInput.focus();
      showToast("Host pattern required", true);
      setTimeout(() => hostInput.classList.remove("error"), 2e3);
      return;
    }
    let injectHeaders;
    let injectQuery;
    try {
      injectHeaders = parseJsonOrEmpty(headersJson);
      injectQuery = parseJsonOrEmpty(queryJson);
    } catch (e) {
      const errorMsg = e instanceof Error ? e.message : String(e);
      showToast(`Invalid JSON: ${errorMsg}`, true);
      return;
    }
    const rule = {
      hostPattern,
      injectHeaders,
      injectQuery
    };
    const next = structuredClone(settings);
    next.injectionRules.push(rule);
    hostInput.value = "";
    headersInput.value = "";
    queryInput.value = "";
    await saveSettings(next);
    showToast("Rule added");
  };
  function switchTab(tab) {
    const requestsPage = document.getElementById("tab-requests");
    const settingsPage = document.getElementById("tab-settings");
    const advancedPage = document.getElementById("tab-advanced");
    const requestsToolbar = document.getElementById("toolbar-requests");
    const settingsToolbar = document.getElementById("toolbar-settings");
    const advancedToolbar = document.getElementById("toolbar-advanced");
    const settingsBtn = document.getElementById("btnSettings");
    requestsPage?.classList.remove("active");
    settingsPage?.classList.remove("active");
    advancedPage?.classList.remove("active");
    requestsToolbar?.classList.add("hidden");
    settingsToolbar?.classList.add("hidden");
    advancedToolbar?.classList.add("hidden");
    settingsBtn?.classList.remove("active");
    if (tab === "settings") {
      settingsPage?.classList.add("active");
      settingsToolbar?.classList.remove("hidden");
      settingsBtn?.classList.add("active");
    } else if (tab === "advanced") {
      advancedPage?.classList.add("active");
      advancedToolbar?.classList.remove("hidden");
      settingsBtn?.classList.add("active");
    } else {
      requestsPage?.classList.add("active");
      requestsToolbar?.classList.remove("hidden");
    }
  }
  qs("btnSettings").onclick = () => switchTab("settings");
  qs("btnBackToRequests").onclick = () => switchTab("requests");
  qs("btnBackToSettings").onclick = () => switchTab("settings");
  qs("advancedLink").onclick = () => switchTab("advanced");
  var port = chrome.runtime.connect({ name: "FRANZAI_SIDEPANEL" });
  port.onMessage.addListener(async (evt) => {
    if (evt.type === BG_EVT.LOGS_UPDATED || evt.type === BG_EVT.SETTINGS_UPDATED) {
      await loadAll();
    }
  });
  document.addEventListener("keydown", (e) => {
    const target = e.target;
    if (target.tagName === "INPUT" || target.tagName === "TEXTAREA" || target.tagName === "SELECT") {
      if (e.key === "Escape") {
        target.blur();
      }
      return;
    }
    if (e.key === "Escape" && selectedLogId) {
      closeDetailPane();
      return;
    }
    if (e.key === "j" || e.key === "k" || e.key === "ArrowDown" || e.key === "ArrowUp") {
      e.preventDefault();
      const filteredLogs = filterLogs(logs);
      const sortedLogs = sortLogs(filteredLogs);
      if (!sortedLogs.length) return;
      const currentIndex = selectedLogId ? sortedLogs.findIndex((l) => l.id === selectedLogId) : -1;
      let newIndex;
      if (e.key === "j" || e.key === "ArrowDown") {
        newIndex = currentIndex < sortedLogs.length - 1 ? currentIndex + 1 : 0;
      } else {
        newIndex = currentIndex > 0 ? currentIndex - 1 : sortedLogs.length - 1;
      }
      const newLog = sortedLogs[newIndex];
      selectedLogId = newLog.id;
      renderDetails(newLog);
      document.querySelectorAll(".item.active").forEach((el) => el.classList.remove("active"));
      const items = logsList.querySelectorAll(".item");
      if (items[newIndex]) {
        items[newIndex].classList.add("active");
        items[newIndex].scrollIntoView({ block: "nearest" });
      }
      const detailPane = document.getElementById("detailPane");
      if (detailPane) detailPane.classList.add("visible");
      return;
    }
    if (e.key === "c" && selectedLogId) {
      const log3 = logs.find((l) => l.id === selectedLogId);
      if (log3) {
        navigator.clipboard.writeText(log3.url);
        showToast("URL copied to clipboard");
      }
      return;
    }
    if (e.key === "/") {
      e.preventDefault();
      const searchInput = document.getElementById("filterSearch");
      if (searchInput) searchInput.focus();
      return;
    }
    if (e.key === "r" && !e.metaKey && !e.ctrlKey) {
      loadAll();
      return;
    }
  });
  document.querySelectorAll(".table-header .sortable").forEach((el) => {
    el.onclick = () => {
      const col = el.getAttribute("data-col");
      if (!col) return;
      if (sortColumn === col) {
        sortDir = sortDir === "asc" ? "desc" : "asc";
      } else {
        sortColumn = col;
        sortDir = col === "ts" ? "desc" : "asc";
      }
      renderLogs();
      saveUIPrefs();
    };
  });
  function initResizableSplit() {
    const layout = document.querySelector(".requests-layout");
    const listPane = document.querySelector(".listPane");
    const detailPane = document.querySelector(".detailPane");
    if (!layout || !listPane || !detailPane) return;
    const resizer = document.createElement("div");
    resizer.className = "split-resizer";
    layout.insertBefore(resizer, detailPane);
    let startY = 0;
    let startHeight = 0;
    const onMouseMove = (e) => {
      const delta = e.clientY - startY;
      const newHeight = Math.max(50, Math.min(startHeight + delta, layout.clientHeight - 100));
      listPane.style.height = newHeight + "px";
      listPane.style.maxHeight = "none";
      listPane.style.flex = "none";
    };
    const onMouseUp = () => {
      document.removeEventListener("mousemove", onMouseMove);
      document.removeEventListener("mouseup", onMouseUp);
      document.body.style.cursor = "";
      document.body.style.userSelect = "";
    };
    resizer.addEventListener("mousedown", (e) => {
      e.preventDefault();
      startY = e.clientY;
      startHeight = listPane.offsetHeight;
      document.body.style.cursor = "row-resize";
      document.body.style.userSelect = "none";
      document.addEventListener("mousemove", onMouseMove);
      document.addEventListener("mouseup", onMouseUp);
    });
  }
  function initResizableColumns() {
    const header = document.querySelector(".table-header");
    if (!header) return;
    const cols = header.querySelectorAll("div");
    cols.forEach((col, index) => {
      if (index === cols.length - 1) return;
      const resizer = document.createElement("div");
      resizer.className = "col-resizer";
      col.style.position = "relative";
      col.appendChild(resizer);
      let startX = 0;
      let startWidth = 0;
      const onMouseMove = (e) => {
        const delta = e.clientX - startX;
        const newWidth = Math.max(30, startWidth + delta);
        updateColumnWidth(index, newWidth);
      };
      const onMouseUp = () => {
        document.removeEventListener("mousemove", onMouseMove);
        document.removeEventListener("mouseup", onMouseUp);
        document.body.style.cursor = "";
        document.body.style.userSelect = "";
      };
      resizer.addEventListener("mousedown", (e) => {
        e.preventDefault();
        e.stopPropagation();
        startX = e.clientX;
        startWidth = col.offsetWidth;
        document.body.style.cursor = "col-resize";
        document.body.style.userSelect = "none";
        document.addEventListener("mousemove", onMouseMove);
        document.addEventListener("mouseup", onMouseUp);
      });
    });
  }
  function updateColumnWidth(index, width) {
    colWidths[index] = width;
    applyColumnWidths();
    saveUIPrefs();
  }
  function applyColumnWidths() {
    const template = colWidths.map((w) => w === -1 ? "1fr" : w + "px").join(" ");
    const header = document.querySelector(".table-header");
    if (header) header.style.gridTemplateColumns = template;
    document.querySelectorAll(".item").forEach((item) => {
      item.style.gridTemplateColumns = template;
    });
  }
  setTimeout(() => {
    initResizableSplit();
    initResizableColumns();
  }, 100);
  var versionEl = document.getElementById("version");
  if (versionEl) versionEl.textContent = `v${BRIDGE_VERSION}`;
  var debouncedRenderLogs = debounce(() => renderLogs(), 150);
  qs("filterSearch").oninput = (e) => {
    filterSearch = e.target.value;
    debouncedRenderLogs();
  };
  qs("filterMethod").onchange = (e) => {
    filterMethod = e.target.value;
    renderLogs();
  };
  qs("filterStatus").onchange = (e) => {
    filterStatus = e.target.value;
    renderLogs();
  };
  var clearFiltersBtn = document.getElementById("btnClearFilters");
  if (clearFiltersBtn) {
    clearFiltersBtn.onclick = clearFilters;
  }
  qs("btnExportLogs").onclick = () => {
    if (!logs.length) {
      showToast("No logs to export", true);
      return;
    }
    const exportData = {
      exportedAt: (/* @__PURE__ */ new Date()).toISOString(),
      version: BRIDGE_VERSION,
      totalLogs: logs.length,
      logs: logs.map((l) => ({
        id: l.id,
        ts: new Date(l.ts).toISOString(),
        method: l.method,
        url: l.url,
        status: l.status,
        statusText: l.statusText,
        elapsedMs: l.elapsedMs,
        error: l.error,
        requestHeaders: l.requestHeaders,
        requestBody: l.requestBodyPreview,
        responseHeaders: l.responseHeaders,
        responseBody: l.responseBodyPreview,
        pageOrigin: l.pageOrigin,
        tabId: l.tabId
      }))
    };
    const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: "application/json" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `franzai-bridge-logs-${(/* @__PURE__ */ new Date()).toISOString().slice(0, 19).replace(/:/g, "-")}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    showToast(`Exported ${logs.length} logs`);
  };
  (async () => {
    await loadUIPrefs();
    applyColumnWidths();
    await loadAll();
  })();
})();
//# sourceMappingURL=sidepanel.js.map
