"use strict";
(() => {
  // src/shared/constants.ts
  var BRIDGE_SOURCE = "FRANZAI_BRIDGE";
  var BRIDGE_VERSION = "2.0.27";
  var BRIDGE_TIMEOUT_MS = 3e4;
  var MAX_BODY_BYTES = 5 * 1024 * 1024;

  // src/shared/messages.ts
  var PAGE_MSG = {
    FETCH_REQUEST: "FETCH_REQUEST",
    FETCH_ABORT: "FETCH_ABORT",
    FETCH_RESPONSE: "FETCH_RESPONSE",
    BRIDGE_READY: "BRIDGE_READY"
  };

  // src/shared/logger.ts
  var LEVELS = {
    debug: 10,
    info: 20,
    warn: 30,
    error: 40,
    silent: 99
  };
  function resolveLevel(level) {
    if (level) return level;
    const raw = globalThis.__FRANZAI_LOG_LEVEL__;
    if (typeof raw === "string" && raw in LEVELS) return raw;
    return "info";
  }
  function createLogger(scope, level, c = console) {
    const chosen = resolveLevel(level);
    const min = LEVELS[chosen];
    const prefix = `[FranzAI Bridge/${scope}]`;
    return {
      debug: (...args) => {
        if (min <= LEVELS.debug) c.debug(prefix, ...args);
      },
      info: (...args) => {
        if (min <= LEVELS.info) c.info(prefix, ...args);
      },
      warn: (...args) => {
        if (min <= LEVELS.warn) c.warn(prefix, ...args);
      },
      error: (...args) => {
        if (min <= LEVELS.error) c.error(prefix, ...args);
      },
      log: (...args) => {
        if (min <= LEVELS.info) c.log(prefix, ...args);
      }
    };
  }

  // src/shared/ids.ts
  function makeId(prefix) {
    if (typeof crypto !== "undefined" && "randomUUID" in crypto) {
      return `${prefix}_${crypto.randomUUID()}`;
    }
    return `${prefix}_${Date.now()}_${Math.random().toString(16).slice(2)}`;
  }

  // src/injected.ts
  var log = createLogger("page");
  var REQUEST_META = Symbol.for("franzaiBridgeMeta");
  var textEncoder = new TextEncoder();
  function normalizeMode(mode) {
    if (mode === "auto" || mode === "always" || mode === "off") return mode;
    return void 0;
  }
  function getBridgeConfig() {
    const w = window;
    const existing = w.__franzaiBridgeConfig;
    const config = existing && typeof existing === "object" ? existing : {};
    const normalized = normalizeMode(config.mode) ?? "always";
    config.mode = normalized;
    w.__franzaiBridgeConfig = config;
    return config;
  }
  var bridgeConfig = getBridgeConfig();
  function setRequestMode(request, mode) {
    if (!mode) return;
    try {
      Object.defineProperty(request, REQUEST_META, {
        value: mode,
        enumerable: false
      });
    } catch {
    }
  }
  function getRequestMode(request) {
    const meta = request;
    return normalizeMode(meta[REQUEST_META]);
  }
  function createAbortError(message) {
    try {
      return new DOMException(message, "AbortError");
    } catch {
      const err = new Error(message);
      err.name = "AbortError";
      return err;
    }
  }
  function isAbortError(error) {
    return error instanceof Error && error.name === "AbortError";
  }
  function resolveUrl(input) {
    if (typeof input === "string") return new URL(input, window.location.href).toString();
    if (input instanceof URL) return new URL(input.toString(), window.location.href).toString();
    return new URL(input.url, window.location.href).toString();
  }
  function headersToLite(headers) {
    if (!headers) return void 0;
    if (headers instanceof Headers) {
      const entries = [];
      headers.forEach((value, key) => entries.push([key, value]));
      return entries;
    }
    if (Array.isArray(headers)) return headers;
    return headers;
  }
  function modeFromInit(init) {
    return normalizeMode(init?.franzai?.mode);
  }
  function resolveBridgeMode(input, init) {
    const initMode = modeFromInit(init);
    if (initMode) return initMode;
    if (input instanceof Request) {
      const requestMode = getRequestMode(input);
      if (requestMode) return requestMode;
    }
    return bridgeConfig.mode ?? "always";
  }
  function enforceMaxBytes(bytes) {
    if (bytes > MAX_BODY_BYTES) {
      throw new Error(`Request body too large (${bytes} bytes). Max is ${MAX_BODY_BYTES} bytes.`);
    }
  }
  function byteLengthOfString(text) {
    return textEncoder.encode(text).byteLength;
  }
  function isTextualContentType(contentType) {
    if (!contentType) return false;
    const ct = contentType.toLowerCase();
    return ct.startsWith("text/") || ct.includes("json") || ct.includes("xml") || ct.includes("x-www-form-urlencoded");
  }
  function maybeSetContentType(headers, value) {
    if (!value) return;
    if (!headers.has("content-type")) headers.set("content-type", value);
  }
  function toUint8Array(buffer) {
    const bytes = new Uint8Array(buffer);
    enforceMaxBytes(bytes.byteLength);
    return bytes;
  }
  async function bodyToPayload(body, headers) {
    if (typeof body === "string") {
      enforceMaxBytes(byteLengthOfString(body));
      return body;
    }
    if (body instanceof URLSearchParams) {
      const text = body.toString();
      maybeSetContentType(headers, "application/x-www-form-urlencoded;charset=UTF-8");
      enforceMaxBytes(byteLengthOfString(text));
      return text;
    }
    if (typeof FormData !== "undefined" && body instanceof FormData) {
      const response = new Response(body);
      maybeSetContentType(headers, response.headers.get("content-type"));
      return toUint8Array(await response.arrayBuffer());
    }
    if (typeof Blob !== "undefined" && body instanceof Blob) {
      if (isTextualContentType(body.type)) {
        const text = await body.text();
        maybeSetContentType(headers, body.type);
        enforceMaxBytes(byteLengthOfString(text));
        return text;
      }
      maybeSetContentType(headers, body.type);
      return toUint8Array(await body.arrayBuffer());
    }
    if (body instanceof ArrayBuffer) {
      return toUint8Array(body);
    }
    if (ArrayBuffer.isView(body)) {
      const bytes = new Uint8Array(body.buffer, body.byteOffset, body.byteLength);
      enforceMaxBytes(bytes.byteLength);
      return bytes;
    }
    if (typeof ReadableStream !== "undefined" && body instanceof ReadableStream) {
      const response = new Response(body);
      return toUint8Array(await response.arrayBuffer());
    }
    throw new Error("FranzAI Bridge cannot forward this body type");
  }
  async function readRequestBody(request, headers) {
    if (!request.body) return void 0;
    const contentType = headers.get("content-type");
    if (isTextualContentType(contentType)) {
      try {
        const text = await request.clone().text();
        enforceMaxBytes(byteLengthOfString(text));
        return text;
      } catch {
      }
    }
    try {
      return toUint8Array(await request.clone().arrayBuffer());
    } catch {
      throw new Error(
        "FranzAI Bridge cannot forward a locked or unreadable request body"
      );
    }
  }
  async function requestToLite(input, init) {
    const baseSignal = init?.signal;
    if (typeof input === "string" || input instanceof URL) {
      const headers2 = new Headers(init?.headers);
      const bodyPayload2 = init?.body != null ? await bodyToPayload(init.body, headers2) : void 0;
      return {
        url: resolveUrl(input),
        init: {
          method: init?.method,
          headers: headersToLite(headers2),
          body: bodyPayload2,
          redirect: init?.redirect,
          credentials: init?.credentials,
          cache: init?.cache,
          referrer: init?.referrer,
          referrerPolicy: init?.referrerPolicy,
          integrity: init?.integrity,
          keepalive: init?.keepalive
        },
        signal: baseSignal ?? void 0
      };
    }
    const req = input;
    const headers = new Headers(init?.headers ?? req.headers);
    const bodyPayload = init?.body != null ? await bodyToPayload(init.body, headers) : await readRequestBody(req, headers);
    return {
      url: resolveUrl(req.url),
      init: {
        method: init?.method ?? req.method,
        headers: headersToLite(headers),
        body: bodyPayload,
        redirect: init?.redirect ?? req.redirect,
        credentials: init?.credentials ?? req.credentials,
        cache: init?.cache ?? req.cache,
        referrer: init?.referrer ?? req.referrer,
        referrerPolicy: init?.referrerPolicy ?? req.referrerPolicy,
        integrity: init?.integrity ?? req.integrity,
        keepalive: init?.keepalive ?? req.keepalive
      },
      signal: baseSignal ?? req.signal
    };
  }
  var nativeFetch = window.franzaiNativeFetch || window.fetch.bind(window);
  window.franzaiNativeFetch = nativeFetch;
  var nativeRequest = window.franzaiNativeRequest || window.Request;
  window.franzaiNativeRequest = nativeRequest;
  function isCrossOrigin(input) {
    try {
      const url = new URL(resolveUrl(input), window.location.href);
      return url.origin !== window.location.origin;
    } catch {
      return false;
    }
  }
  function shouldUseBridgeForRequest(input, init) {
    const mode = resolveBridgeMode(input, init);
    if (mode === "off") return false;
    if (mode === "always") return true;
    return isCrossOrigin(input);
  }
  var franzai = {
    version: BRIDGE_VERSION,
    async ping() {
      return { ok: true, version: franzai.version };
    },
    setMode(mode) {
      bridgeConfig.mode = normalizeMode(mode) ?? "auto";
      return bridgeConfig.mode;
    },
    getMode() {
      return bridgeConfig.mode ?? "auto";
    },
    async fetch(input, init) {
      const lite = await requestToLite(input, init);
      if (lite.signal?.aborted) {
        throw createAbortError("The operation was aborted");
      }
      const requestId = makeId("req");
      const req = { requestId, url: lite.url, init: lite.init };
      const resp = await new Promise((resolve, reject) => {
        let done = false;
        const cleanup = () => {
          window.removeEventListener("message", onMessage);
          if (lite.signal) lite.signal.removeEventListener("abort", onAbort);
          clearTimeout(timeoutId);
        };
        const finishResolve = (value) => {
          if (done) return;
          done = true;
          cleanup();
          resolve(value);
        };
        const finishReject = (error) => {
          if (done) return;
          done = true;
          cleanup();
          reject(error);
        };
        const onAbort = () => {
          window.postMessage(
            {
              source: BRIDGE_SOURCE,
              type: PAGE_MSG.FETCH_ABORT,
              payload: { requestId }
            },
            "*"
          );
          finishReject(createAbortError("The operation was aborted"));
        };
        const onMessage = (ev) => {
          if (ev.source !== window) return;
          const data = ev.data;
          if (!data || data.source !== BRIDGE_SOURCE) return;
          if (data.type !== PAGE_MSG.FETCH_RESPONSE) return;
          const payload = data.payload;
          const responseObj = payload?.response;
          if (!responseObj || responseObj.requestId !== requestId) return;
          finishResolve(payload);
        };
        const timeoutId = window.setTimeout(() => {
          finishResolve({
            ok: false,
            error: `Timed out waiting for FranzAI Bridge response after ${BRIDGE_TIMEOUT_MS}ms. Check that the extension is installed, enabled, and that this origin is allowed.`
          });
        }, BRIDGE_TIMEOUT_MS);
        window.addEventListener("message", onMessage);
        if (lite.signal) lite.signal.addEventListener("abort", onAbort, { once: true });
        window.postMessage(
          {
            source: BRIDGE_SOURCE,
            type: PAGE_MSG.FETCH_REQUEST,
            payload: req
          },
          "*"
        );
      });
      if (!resp.ok || !resp.response) {
        const msg = resp.error ?? resp.response?.error ?? "Unknown error";
        log.warn("Bridge fetch failed", msg);
        throw new Error(`FranzAI Bridge fetch failed: ${msg}`);
      }
      const r = resp.response;
      return new Response(r.bodyText, {
        status: r.status,
        statusText: r.statusText,
        headers: r.headers
      });
    }
  };
  async function hookedFetch(input, init) {
    const mode = resolveBridgeMode(input, init);
    if (!shouldUseBridgeForRequest(input, init)) {
      return nativeFetch(input, init);
    }
    try {
      return await franzai.fetch(input, init);
    } catch (e) {
      if (isAbortError(e)) throw e;
      if (mode === "always") throw e;
      const msg = e instanceof Error ? e.message : String(e);
      log.debug("Bridge fetch failed, falling back to native fetch", msg);
      try {
        return await nativeFetch(input, init);
      } catch {
        throw new Error(msg);
      }
    }
  }
  function installRequestHook() {
    const w = window;
    if (w.__franzaiRequestHookInstalled) return;
    w.__franzaiRequestHookInstalled = true;
    const FranzaiRequest = function(input, init) {
      const req = new nativeRequest(input, init);
      setRequestMode(req, modeFromInit(init));
      return req;
    };
    FranzaiRequest.prototype = nativeRequest.prototype;
    Object.setPrototypeOf(FranzaiRequest, nativeRequest);
    try {
      Object.defineProperty(window, "Request", {
        configurable: true,
        writable: true,
        value: FranzaiRequest
      });
    } catch {
      window.Request = FranzaiRequest;
    }
  }
  window.franzai = franzai;
  (() => {
    const w = window;
    if (w.__franzaiFetchHookInstalled) return;
    w.__franzaiFetchHookInstalled = true;
    installRequestHook();
    try {
      Object.defineProperty(window, "fetch", {
        configurable: true,
        writable: true,
        value: hookedFetch
      });
    } catch {
      window.fetch = hookedFetch;
    }
    window.postMessage(
      {
        source: BRIDGE_SOURCE,
        type: PAGE_MSG.BRIDGE_READY,
        payload: { version: franzai.version }
      },
      "*"
    );
    log.info("window.fetch + window.Request are routed through FranzAI Bridge");
  })();
})();
//# sourceMappingURL=injected.js.map
