"use strict";
(() => {
  // src/shared/constants.ts
  var BRIDGE_SOURCE = "FRANZAI_BRIDGE";
  var BRIDGE_TIMEOUT_MS = 3e4;
  var MAX_BODY_BYTES = 5 * 1024 * 1024;
  var RUNTIME_MESSAGE_TIMEOUT_MS = 15e3;

  // src/shared/messages.ts
  var BG_MSG = {
    FETCH: "FRANZAI_FETCH",
    FETCH_ABORT: "FRANZAI_FETCH_ABORT",
    GET_SETTINGS: "FRANZAI_GET_SETTINGS",
    SET_SETTINGS: "FRANZAI_SET_SETTINGS",
    GET_LOGS: "FRANZAI_GET_LOGS",
    CLEAR_LOGS: "FRANZAI_CLEAR_LOGS"
  };
  var PAGE_MSG = {
    FETCH_REQUEST: "FETCH_REQUEST",
    FETCH_ABORT: "FETCH_ABORT",
    FETCH_RESPONSE: "FETCH_RESPONSE",
    BRIDGE_READY: "BRIDGE_READY"
  };

  // src/shared/logger.ts
  var LEVELS = {
    debug: 10,
    info: 20,
    warn: 30,
    error: 40,
    silent: 99
  };
  function resolveLevel(level) {
    if (level) return level;
    const raw = globalThis.__FRANZAI_LOG_LEVEL__;
    if (typeof raw === "string" && raw in LEVELS) return raw;
    return "info";
  }
  function createLogger(scope, level, c = console) {
    const chosen = resolveLevel(level);
    const min = LEVELS[chosen];
    const prefix = `[FranzAI Bridge/${scope}]`;
    return {
      debug: (...args) => {
        if (min <= LEVELS.debug) c.debug(prefix, ...args);
      },
      info: (...args) => {
        if (min <= LEVELS.info) c.info(prefix, ...args);
      },
      warn: (...args) => {
        if (min <= LEVELS.warn) c.warn(prefix, ...args);
      },
      error: (...args) => {
        if (min <= LEVELS.error) c.error(prefix, ...args);
      },
      log: (...args) => {
        if (min <= LEVELS.info) c.log(prefix, ...args);
      }
    };
  }

  // src/shared/runtime.ts
  var log = createLogger("runtime");
  function sendRuntimeMessage(message, options = {}) {
    const timeoutMs = options.timeoutMs ?? RUNTIME_MESSAGE_TIMEOUT_MS;
    return new Promise((resolve, reject) => {
      let done = false;
      const timeoutId = window.setTimeout(() => {
        if (done) return;
        done = true;
        reject(new Error(`Timeout waiting for runtime response after ${timeoutMs}ms`));
      }, timeoutMs);
      try {
        chrome.runtime.sendMessage(message, (resp) => {
          if (done) return;
          done = true;
          clearTimeout(timeoutId);
          const err = chrome.runtime.lastError;
          if (err) {
            log.warn("sendMessage failed", err.message || err);
            reject(err);
            return;
          }
          resolve(resp);
        });
      } catch (e) {
        if (done) return;
        done = true;
        clearTimeout(timeoutId);
        log.error("sendMessage threw", e);
        reject(e);
      }
    });
  }

  // src/contentScript.ts
  var log2 = createLogger("content");
  var injected = false;
  function injectPageScript() {
    if (injected) return;
    injected = true;
    try {
      const script = document.createElement("script");
      script.src = chrome.runtime.getURL("injected.js");
      script.async = false;
      (document.documentElement || document.head || document.body).appendChild(script);
      script.onload = () => script.remove();
    } catch (e) {
      log2.error("Failed to inject page script", e);
    }
  }
  injectPageScript();
  window.addEventListener("message", async (event) => {
    if (event.source !== window) return;
    const data = event.data;
    if (!data || data.source !== BRIDGE_SOURCE) return;
    if (data.type === PAGE_MSG.BRIDGE_READY) {
      log2.info("Bridge ready", data.payload?.version);
      return;
    }
    if (data.type === PAGE_MSG.FETCH_ABORT) {
      const requestId = data.payload?.requestId;
      if (!requestId) return;
      try {
        await sendRuntimeMessage({
          type: BG_MSG.FETCH_ABORT,
          payload: { requestId }
        });
      } catch (e) {
        log2.warn("Failed to forward abort", e);
      }
      return;
    }
    if (data.type === PAGE_MSG.FETCH_REQUEST) {
      const req = data.payload;
      if (!req?.requestId) {
        log2.warn("Dropping request without requestId");
        return;
      }
      const payload = {
        ...req,
        pageOrigin: window.location.origin
      };
      try {
        const resp = await sendRuntimeMessage({
          type: BG_MSG.FETCH,
          payload
        }, {
          timeoutMs: BRIDGE_TIMEOUT_MS + 5e3
        });
        window.postMessage(
          {
            source: BRIDGE_SOURCE,
            type: PAGE_MSG.FETCH_RESPONSE,
            payload: resp
          },
          "*"
        );
      } catch (error) {
        const message = error instanceof Error ? error.message : String(error);
        const errorResponse = {
          requestId: req.requestId,
          ok: false,
          status: 0,
          statusText: "Bridge Error",
          headers: {},
          bodyText: "",
          elapsedMs: 0,
          error: message
        };
        log2.error("Bridge error reaching background", message);
        window.postMessage(
          {
            source: BRIDGE_SOURCE,
            type: PAGE_MSG.FETCH_RESPONSE,
            payload: {
              ok: false,
              response: errorResponse,
              error: `Failed to reach FranzAI Bridge background: ${message}`
            }
          },
          "*"
        );
      }
    }
  });
})();
//# sourceMappingURL=contentScript.js.map
