"use strict";
(() => {
  // src/shared/messages.ts
  var BG_MSG = {
    FETCH: "FRANZAI_FETCH",
    FETCH_ABORT: "FRANZAI_FETCH_ABORT",
    GET_SETTINGS: "FRANZAI_GET_SETTINGS",
    SET_SETTINGS: "FRANZAI_SET_SETTINGS",
    GET_LOGS: "FRANZAI_GET_LOGS",
    CLEAR_LOGS: "FRANZAI_CLEAR_LOGS"
  };
  var BG_EVT = {
    LOGS_UPDATED: "FRANZAI_LOGS_UPDATED",
    SETTINGS_UPDATED: "FRANZAI_SETTINGS_UPDATED"
  };

  // src/shared/defaults.ts
  var SETTINGS_VERSION = 3;
  var DEFAULT_SETTINGS = {
    settingsVersion: SETTINGS_VERSION,
    allowedOrigins: [
      "*"
      // Allow ALL origins by default - this extension is for circumventing CORS!
    ],
    allowedDestinations: [
      "*"
      // Allow ALL destinations by default - this extension is for circumventing CORS!
    ],
    env: {
      OPENAI_API_KEY: "",
      ANTHROPIC_API_KEY: "",
      GOOGLE_API_KEY: "",
      MISTRAL_API_KEY: ""
    },
    injectionRules: [],
    maxLogs: 200
  };

  // src/shared/constants.ts
  var FETCH_TIMEOUT_MS = 25e3;
  var MAX_BODY_BYTES = 5 * 1024 * 1024;
  var REQUEST_BODY_PREVIEW_LIMIT = 25e3;
  var RESPONSE_BODY_PREVIEW_LIMIT = 5e4;
  var MIN_LOGS_LIMIT = 10;
  var MAX_LOGS_LIMIT = 1e3;

  // src/shared/normalize.ts
  function toStringArray(value, fallback) {
    if (!Array.isArray(value)) return fallback;
    return value.filter((item) => typeof item === "string");
  }
  function toStringDict(value) {
    const out = {};
    if (!value || typeof value !== "object") return out;
    for (const [k, v] of Object.entries(value)) {
      if (typeof v === "string") out[k] = v;
    }
    return out;
  }
  function toRules(value) {
    if (!Array.isArray(value)) return [];
    const rules = [];
    for (const item of value) {
      if (!item || typeof item !== "object") continue;
      const rule = item;
      if (typeof rule.hostPattern !== "string") continue;
      rules.push({
        hostPattern: rule.hostPattern,
        injectHeaders: rule.injectHeaders ? toStringDict(rule.injectHeaders) : void 0,
        injectQuery: rule.injectQuery ? toStringDict(rule.injectQuery) : void 0
      });
    }
    return rules;
  }
  function clampNumber(value, fallback, min, max) {
    if (typeof value !== "number" || Number.isNaN(value)) return fallback;
    return Math.max(min, Math.min(max, Math.floor(value)));
  }
  function normalizeSettings(input) {
    const allowedOrigins = toStringArray(input?.allowedOrigins, DEFAULT_SETTINGS.allowedOrigins);
    const allowedDestinations = toStringArray(
      input?.allowedDestinations,
      DEFAULT_SETTINGS.allowedDestinations
    );
    const env = {
      ...DEFAULT_SETTINGS.env,
      ...toStringDict(input?.env)
    };
    const injectionRules = toRules(input?.injectionRules);
    const maxLogs = clampNumber(
      input?.maxLogs,
      DEFAULT_SETTINGS.maxLogs,
      MIN_LOGS_LIMIT,
      MAX_LOGS_LIMIT
    );
    return {
      settingsVersion: SETTINGS_VERSION,
      allowedOrigins,
      allowedDestinations,
      env,
      injectionRules,
      maxLogs
    };
  }

  // src/shared/logger.ts
  var LEVELS = {
    debug: 10,
    info: 20,
    warn: 30,
    error: 40,
    silent: 99
  };
  function resolveLevel(level) {
    if (level) return level;
    const raw = globalThis.__FRANZAI_LOG_LEVEL__;
    if (typeof raw === "string" && raw in LEVELS) return raw;
    return "info";
  }
  function createLogger(scope, level, c = console) {
    const chosen = resolveLevel(level);
    const min = LEVELS[chosen];
    const prefix = `[FranzAI Bridge/${scope}]`;
    return {
      debug: (...args) => {
        if (min <= LEVELS.debug) c.debug(prefix, ...args);
      },
      info: (...args) => {
        if (min <= LEVELS.info) c.info(prefix, ...args);
      },
      warn: (...args) => {
        if (min <= LEVELS.warn) c.warn(prefix, ...args);
      },
      error: (...args) => {
        if (min <= LEVELS.error) c.error(prefix, ...args);
      },
      log: (...args) => {
        if (min <= LEVELS.info) c.log(prefix, ...args);
      }
    };
  }

  // src/shared/storage.ts
  var log = createLogger("storage");
  var SETTINGS_KEY = "franzaiSettings";
  var LOGS_KEY = "franzaiLogs";
  function sessionStorageOrLocal() {
    const anyChrome = chrome;
    return anyChrome.storage?.session ?? chrome.storage.local;
  }
  function migrateSettings(oldSettings) {
    const newSettings = structuredClone(DEFAULT_SETTINGS);
    if (oldSettings?.env) {
      for (const [key, value] of Object.entries(oldSettings.env)) {
        if (value && value.trim() !== "") {
          newSettings.env[key] = value;
        }
      }
    }
    if (oldSettings?.injectionRules && oldSettings.injectionRules.length > 0) {
      newSettings.injectionRules = oldSettings.injectionRules;
    }
    log.info("Settings migrated to version", SETTINGS_VERSION, "preserving ENV vars and rules");
    return newSettings;
  }
  async function getSettings() {
    const data = await chrome.storage.local.get(SETTINGS_KEY);
    const stored = data[SETTINGS_KEY];
    const storedVersion = stored?.settingsVersion ?? 0;
    if (storedVersion < SETTINGS_VERSION) {
      log.info("Settings version outdated:", storedVersion, "\u2192", SETTINGS_VERSION, "- migrating");
      const migrated = migrateSettings(stored);
      await setSettings(migrated);
      return migrated;
    }
    return normalizeSettings(stored);
  }
  async function setSettings(settings) {
    const normalized = normalizeSettings(settings);
    await chrome.storage.local.set({ [SETTINGS_KEY]: normalized });
  }
  async function getLogs() {
    const store = sessionStorageOrLocal();
    const data = await store.get(LOGS_KEY);
    return data[LOGS_KEY] ?? [];
  }
  async function setLogs(logs) {
    const store = sessionStorageOrLocal();
    await store.set({ [LOGS_KEY]: logs });
  }
  async function appendLog(entry, maxLogs) {
    const logs = await getLogs();
    logs.unshift(entry);
    logs.length = Math.min(logs.length, maxLogs);
    await setLogs(logs);
  }
  async function clearLogs() {
    const store = sessionStorageOrLocal();
    await store.remove(LOGS_KEY);
  }

  // src/shared/wildcard.ts
  function wildcardToRegExp(pattern) {
    const escaped = pattern.trim().replace(/[.+^${}()|[\]\\]/g, "\\$&").replace(/\*/g, ".*");
    return new RegExp(`^${escaped}$`, "i");
  }
  function matchesAnyPattern(value, patterns) {
    const v = value.trim();
    return patterns.some((p) => wildcardToRegExp(p).test(v));
  }

  // src/shared/providers.ts
  function builtinProviderRules() {
    return [
      {
        hostPattern: "api.openai.com",
        injectHeaders: {
          Authorization: "Bearer ${OPENAI_API_KEY}"
        }
      },
      {
        hostPattern: "api.anthropic.com",
        injectHeaders: {
          "x-api-key": "${ANTHROPIC_API_KEY}",
          "anthropic-version": "2023-06-01"
          // Hardcoded - users don't need to configure this
        }
      },
      {
        hostPattern: "generativelanguage.googleapis.com",
        injectHeaders: {
          "x-goog-api-key": "${GOOGLE_API_KEY}"
        }
      },
      {
        hostPattern: "api.mistral.ai",
        injectHeaders: {
          Authorization: "Bearer ${MISTRAL_API_KEY}"
        }
      }
    ];
  }
  function expandTemplate(input, env) {
    return input.replace(/\$\{([A-Z0-9_]+)\}/g, (_, name) => env[name] ?? "");
  }
  function headersToObject(headers) {
    const out = {};
    if (!headers) return out;
    if (headers instanceof Headers) {
      headers.forEach((value, key) => {
        out[key] = value;
      });
      return out;
    }
    if (Array.isArray(headers)) {
      for (const [k, v] of headers) out[k] = v;
      return out;
    }
    for (const [k, v] of Object.entries(headers)) {
      out[k] = String(v);
    }
    return out;
  }
  function hasHeader(headers, name) {
    const n = name.toLowerCase();
    return Object.keys(headers).some((k) => k.toLowerCase() === n);
  }

  // src/shared/ids.ts
  function makeId(prefix) {
    if (typeof crypto !== "undefined" && "randomUUID" in crypto) {
      return `${prefix}_${crypto.randomUUID()}`;
    }
    return `${prefix}_${Date.now()}_${Math.random().toString(16).slice(2)}`;
  }

  // src/background.ts
  var log2 = createLogger("bg");
  var ports = /* @__PURE__ */ new Set();
  var inFlight = /* @__PURE__ */ new Map();
  var abortedByPage = /* @__PURE__ */ new Set();
  function broadcast(evt) {
    for (const port of ports) {
      try {
        port.postMessage(evt);
      } catch {
      }
    }
  }
  function isDestinationAllowed(url, allowedDestinations) {
    const full = url.toString();
    const host = url.hostname;
    for (const pat of allowedDestinations) {
      const p = pat.trim();
      if (!p) continue;
      if (p.includes("://")) {
        if (wildcardToRegExp(p).test(full)) return true;
      } else {
        if (wildcardToRegExp(p).test(host)) return true;
      }
    }
    return false;
  }
  function applyInjectionRules(args) {
    const { url, headers, env, rules } = args;
    for (const rule of rules) {
      const hostRe = wildcardToRegExp(rule.hostPattern);
      if (!hostRe.test(url.hostname)) continue;
      if (rule.injectHeaders) {
        for (const [hk, hvTemplate] of Object.entries(rule.injectHeaders)) {
          if (hasHeader(headers, hk)) continue;
          const value = expandTemplate(hvTemplate, env).trim();
          if (value) headers[hk] = value;
        }
      }
      if (rule.injectQuery) {
        for (const [qk, qvTemplate] of Object.entries(rule.injectQuery)) {
          if (url.searchParams.has(qk)) continue;
          const value = expandTemplate(qvTemplate, env).trim();
          if (value) url.searchParams.set(qk, value);
        }
      }
    }
  }
  function previewBody(body, max) {
    if (body == null) return "";
    if (body instanceof Uint8Array) return `[binary body ${body.byteLength} bytes]`;
    if (body instanceof ArrayBuffer) return `[binary body ${body.byteLength} bytes]`;
    if (typeof body !== "string") return `[${typeof body} body omitted]`;
    if (body.length <= max) return body;
    return body.slice(0, max) + `

...[truncated, total ${body.length} chars]`;
  }
  function makeErrorResponse(requestId, statusText, message, elapsedMs) {
    return {
      requestId,
      ok: false,
      status: 0,
      statusText,
      headers: {},
      bodyText: "",
      elapsedMs,
      error: message
    };
  }
  async function finalizeWithError(args) {
    const { requestId, statusText, message, started, logEntry, maxLogs } = args;
    const elapsedMs = Date.now() - started;
    logEntry.status = 0;
    logEntry.statusText = statusText;
    logEntry.error = message;
    logEntry.elapsedMs = elapsedMs;
    await appendLog(logEntry, maxLogs);
    broadcast({ type: BG_EVT.LOGS_UPDATED });
    const response = makeErrorResponse(requestId, statusText, message, elapsedMs);
    return { ok: false, response, error: message };
  }
  chrome.runtime.onInstalled.addListener(async () => {
    try {
      log2.info("onInstalled: ensuring settings in storage");
      const settings = await getSettings();
      await setSettings(settings);
    } catch (e) {
      log2.error("onInstalled failed", e);
    }
  });
  chrome.runtime.onConnect.addListener((port) => {
    if (port.name !== "FRANZAI_SIDEPANEL") return;
    ports.add(port);
    port.onDisconnect.addListener(() => ports.delete(port));
  });
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    (async () => {
      try {
        switch (msg.type) {
          case BG_MSG.GET_SETTINGS: {
            const settings = await getSettings();
            sendResponse({ ok: true, settings });
            return;
          }
          case BG_MSG.SET_SETTINGS: {
            await setSettings(msg.payload);
            broadcast({ type: BG_EVT.SETTINGS_UPDATED });
            sendResponse({ ok: true });
            return;
          }
          case BG_MSG.GET_LOGS: {
            const logs = await getLogs();
            sendResponse({ ok: true, logs });
            return;
          }
          case BG_MSG.CLEAR_LOGS: {
            await clearLogs();
            broadcast({ type: BG_EVT.LOGS_UPDATED });
            sendResponse({ ok: true });
            return;
          }
          case BG_MSG.FETCH_ABORT: {
            const requestId = msg.payload?.requestId;
            if (requestId) {
              abortedByPage.add(requestId);
              const controller = inFlight.get(requestId);
              if (controller) controller.abort();
            }
            sendResponse({ ok: true });
            return;
          }
          case BG_MSG.FETCH: {
            const settings = await getSettings();
            const payload = msg.payload;
            const tabId = sender.tab?.id;
            const started = Date.now();
            log2.info("FRANZAI_FETCH", {
              requestId: payload.requestId,
              tabId,
              pageOrigin: payload.pageOrigin,
              url: payload.url
            });
            const init = payload.init ?? {};
            const method = (init.method ?? "GET").toUpperCase();
            const requestHeaders = headersToObject(init.headers);
            const logEntry = {
              id: makeId("log"),
              requestId: payload.requestId,
              ts: Date.now(),
              tabId,
              pageOrigin: payload.pageOrigin,
              url: payload.url,
              method,
              requestHeaders,
              requestBodyPreview: previewBody(init.body, REQUEST_BODY_PREVIEW_LIMIT)
            };
            if (!payload.pageOrigin || !matchesAnyPattern(payload.pageOrigin, settings.allowedOrigins)) {
              const env = await finalizeWithError({
                requestId: payload.requestId,
                statusText: "Blocked",
                message: `Blocked: page origin not allowed (${payload.pageOrigin}).`,
                started,
                logEntry,
                maxLogs: settings.maxLogs
              });
              log2.warn("Blocked fetch: origin not allowed", payload.pageOrigin);
              sendResponse(env);
              return;
            }
            let url;
            try {
              url = new URL(payload.url);
            } catch {
              const env = await finalizeWithError({
                requestId: payload.requestId,
                statusText: "Bad URL",
                message: `Invalid URL: ${payload.url}`,
                started,
                logEntry,
                maxLogs: settings.maxLogs
              });
              log2.warn("Blocked fetch: invalid URL", payload.url);
              sendResponse(env);
              return;
            }
            logEntry.url = url.toString();
            if (!isDestinationAllowed(url, settings.allowedDestinations)) {
              const env = await finalizeWithError({
                requestId: payload.requestId,
                statusText: "Blocked",
                message: `Blocked: destination not allowed (${url.hostname}).`,
                started,
                logEntry,
                maxLogs: settings.maxLogs
              });
              log2.warn("Blocked fetch: destination not allowed", url.hostname);
              sendResponse(env);
              return;
            }
            const allRules = [...builtinProviderRules(), ...settings.injectionRules];
            applyInjectionRules({
              url,
              headers: requestHeaders,
              env: settings.env,
              rules: allRules
            });
            const fetchInit = {
              method,
              headers: requestHeaders,
              body: init.body,
              redirect: init.redirect,
              credentials: init.credentials,
              cache: init.cache,
              referrer: init.referrer,
              referrerPolicy: init.referrerPolicy,
              integrity: init.integrity,
              keepalive: init.keepalive
            };
            const controller = new AbortController();
            let timedOut = false;
            const timeoutId = setTimeout(() => {
              timedOut = true;
              controller.abort();
            }, FETCH_TIMEOUT_MS);
            inFlight.set(payload.requestId, controller);
            try {
              const res = await fetch(url.toString(), { ...fetchInit, signal: controller.signal });
              const headersObj = {};
              res.headers.forEach((value, key) => {
                headersObj[key] = value;
              });
              const bodyText = await res.text();
              const elapsedMs = Date.now() - started;
              logEntry.status = res.status;
              logEntry.statusText = res.statusText;
              logEntry.responseHeaders = headersObj;
              logEntry.responseBodyPreview = previewBody(bodyText, RESPONSE_BODY_PREVIEW_LIMIT);
              logEntry.elapsedMs = elapsedMs;
              await appendLog(logEntry, settings.maxLogs);
              broadcast({ type: BG_EVT.LOGS_UPDATED });
              const responseToPage = {
                requestId: payload.requestId,
                ok: res.ok,
                status: res.status,
                statusText: res.statusText,
                headers: headersObj,
                bodyText,
                elapsedMs
              };
              log2.info("Fetch completed", {
                requestId: payload.requestId,
                url: url.toString(),
                status: res.status,
                elapsedMs
              });
              sendResponse({ ok: true, response: responseToPage });
              return;
            } catch (e) {
              const elapsedMs = Date.now() - started;
              const err = e instanceof Error ? e.message : String(e);
              const isAbort = e instanceof Error && e.name === "AbortError";
              let message = err;
              let statusText = "Network Error";
              if (isAbort && timedOut) {
                statusText = "Timeout";
                message = `Timed out after ${FETCH_TIMEOUT_MS}ms`;
              } else if (isAbort && abortedByPage.has(payload.requestId)) {
                statusText = "Aborted";
                message = "Aborted by caller";
              }
              logEntry.error = message;
              logEntry.statusText = statusText;
              logEntry.elapsedMs = elapsedMs;
              await appendLog(logEntry, settings.maxLogs);
              broadcast({ type: BG_EVT.LOGS_UPDATED });
              const responseToPage = {
                requestId: payload.requestId,
                ok: false,
                status: 0,
                statusText,
                headers: {},
                bodyText: "",
                elapsedMs,
                error: message
              };
              log2.error("Fetch failed", { requestId: payload.requestId, statusText, message });
              sendResponse({ ok: false, response: responseToPage, error: message });
              return;
            } finally {
              clearTimeout(timeoutId);
              inFlight.delete(payload.requestId);
              abortedByPage.delete(payload.requestId);
            }
          }
          default: {
            const unknown = msg.type;
            log2.warn("Unknown message type", unknown);
            sendResponse({ ok: false, error: "Unknown message type" });
            return;
          }
        }
      } catch (e) {
        const message = e instanceof Error ? e.message : String(e);
        log2.error("Unhandled error in onMessage", e);
        sendResponse({ ok: false, error: `Internal error: ${message}` });
      }
    })();
    return true;
  });
  chrome.action.onClicked.addListener(async (tab) => {
    log2.info("Extension icon clicked, opening sidepanel for tab", tab.id);
    if (tab.id) {
      try {
        await chrome.sidePanel.open({ tabId: tab.id });
      } catch (e) {
        log2.error("Failed to open sidepanel", e);
      }
    }
  });
  chrome.runtime.onInstalled.addListener(async (details) => {
    log2.info("Extension installed/updated:", details.reason);
    await chrome.sidePanel.setOptions({
      enabled: true
    });
    if (details.reason === "install") {
      try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab?.id) {
          log2.info("Opening sidepanel after install for tab", tab.id);
          await chrome.sidePanel.open({ tabId: tab.id });
        }
      } catch (e) {
        log2.warn("Could not auto-open sidepanel after install", e);
      }
    }
  });
})();
//# sourceMappingURL=background.js.map
